<?php

namespace App\Http\Controllers;

use App\Http\Controllers\CommonController;
use App\Models\Appraisal;
use App\Models\Appraisal_submit;
use App\Models\Book;
use App\Models\BookIssue;
use App\Models\Chat;
use App\Models\Classes;
use App\Models\Club;
use App\Models\ClubMember;
use App\Models\ClubNotice;
use App\Models\DailyAttendances;
use App\Models\Enrollment;
use App\Models\ExamCategory;
use App\Models\FrontendEvent;
use App\Models\Grade;
use App\Models\Hostel;
use App\Models\HostelApplication;
use App\Models\HostelFee;
use App\Models\HostelRoom;
use App\Models\HostelRoomAllocation;
use App\Models\MessageThrade;
use App\Models\Noticeboard;
use App\Models\Routine;
use App\Models\Section;
use App\Models\StudentFeeManager;
use App\Models\Subject;
use App\Models\Syllabus;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;

class StudentController extends Controller
{
    /**
     * Show the student dashboard.
     *
     * @return \Illuminate\Contracts\Support\Renderable
     */
    public function studentDashboard()
    {

        if (auth()->user()->role_id == 7) {
            return view('student.dashboard');
        } else {
            redirect()->route('login')
                ->with('error', 'You are not logged in.');
        }
    }

    /**
     * Show the teacher list.
     *
     * @return \Illuminate\Contracts\Support\Renderable
     */
    public function teacherList(Request $request)
    {
        $search = $request['search'] ?? "";

        if ($search != "") {

            $teachers = User::where(function ($query) use ($search) {
                $query->where('name', 'LIKE', "%{$search}%")
                    ->where('school_id', auth()->user()->school_id)
                    ->where('role_id', 3);
            })->paginate(10);
        } else {
            $teachers = User::where('role_id', 3)->where('school_id', auth()->user()->school_id)->paginate(10);
        }

        return view('student.teacher.teacher_list', compact('teachers', 'search'));
    }

    /**
     * Show the daily attendance.
     *
     * @return \Illuminate\Contracts\Support\Renderable
     */
    public function dailyAttendance(Request $request)
    {
        if (! empty($request->all())) {
            $data                         = $request->all();
            $date                         = '01 ' . $data['month'] . ' ' . $data['year'];
            $page_data['attendance_date'] = strtotime($date);
            $page_data['month']           = $data['month'];
            $page_data['year']            = $data['year'];

            $student_data = (new CommonController)->get_student_details_by_id(auth()->user()->id);
            $classes      = Classes::where('school_id', auth()->user()->school_id)->get();
            $sections     = Section::where(['class_id' => $student_data['class_id']])->get();

            return view('student.attendance.daily_attendance', ['student_data' => $student_data, 'classes' => $classes, 'sections' => $sections, 'page_data' => $page_data]);
        } else {

            $date                         = '01 ' . date('M') . ' ' . date('Y');
            $page_data['attendance_date'] = strtotime($date);
            $page_data['month']           = date('M');
            $page_data['year']            = date('Y');

            $student_data = (new CommonController)->get_student_details_by_id(auth()->user()->id);
            $classes      = Classes::where('school_id', auth()->user()->school_id)->get();
            $sections     = Section::where(['class_id' => $student_data['class_id']])->get();
            return view('student.attendance.daily_attendance', ['student_data' => $student_data, 'classes' => $classes, 'sections' => $sections, 'page_data' => $page_data]);
        }
    }

    public function dailyAttendanceFilter_csv(Request $request)
    {

        $data = $request->all();

        $store_get_data = array_keys($data);

        $data['month']   = substr($store_get_data[0], 0, 3);
        $data['year']    = substr($store_get_data[0], 4, 4);
        $data['role_id'] = substr($store_get_data[0], 9, 5);

        $active_session = get_school_settings(auth()->user()->school_id)->value('running_session');

        $date = '01 ' . $data['month'] . ' ' . $data['year'];

        $first_date = strtotime($date);

        $last_date = date("Y-m-t", strtotime($date));
        $last_date = strtotime($last_date);

        $page_data['month']           = $data['month'];
        $page_data['year']            = $data['year'];
        $page_data['attendance_date'] = $first_date;
        $no_of_users                  = 0;

        $no_of_users            = DailyAttendances::whereBetween('timestamp', [$first_date, $last_date])->where(['school_id' => auth()->user()->school_id, 'session_id' => $active_session])->distinct()->count('student_id');
        $attendance_of_students = DailyAttendances::whereBetween('timestamp', [$first_date, $last_date])->where(['school_id' => auth()->user()->school_id, 'student_id' => auth()->user()->id, 'session_id' => $active_session])->get()->toArray();

        $csv_content    = "Student" . "/" . get_phrase('Date');
        $number_of_days = date('m', $page_data['attendance_date']) == 2 ? (date('Y', $page_data['attendance_date']) % 4 ? 28 : (date('m', $page_data['attendance_date']) % 100 ? 29 : (date('m', $page_data['attendance_date']) % 400 ? 28 : 29))) : ((date('m', $page_data['attendance_date']) - 1) % 7 % 2 ? 30 : 31);
        for ($i = 1; $i <= $number_of_days; $i++) {
            $csv_content .= ',' . get_phrase($i);
        }

        $file = "Attendence_report.csv";

        $student_id_count = 0;

        foreach (array_slice($attendance_of_students, 0, $no_of_users) as $attendance_of_student) {
            $csv_content .= "\n";

            $user_details = (new CommonController)->get_user_by_id_from_user_table($attendance_of_student['student_id']);
            if (date('m', $page_data['attendance_date']) == date('m', $attendance_of_student['timestamp'])) {

                if ($student_id_count != $attendance_of_student['student_id']) {

                    $csv_content .= $user_details['name'] . ',';

                    for ($i = 1; $i <= $number_of_days; $i++) {

                        $page_data['date'] = $i . ' ' . $page_data['month'] . ' ' . $page_data['year'];
                        $timestamp         = strtotime($page_data['date']);

                        $attendance_by_id = DailyAttendances::where(['student_id' => $attendance_of_student['student_id'], 'school_id' => auth()->user()->school_id, 'timestamp' => $timestamp])->first();

                        if (isset($attendance_by_id->status) && $attendance_by_id->status == 1) {
                            $csv_content .= "P,";
                        } elseif (isset($attendance_by_id->status) && $attendance_by_id->status == 0) {
                            $csv_content .= "A,";
                        } else {
                            $csv_content .= ",";
                        }

                        if ($i == $number_of_days) {
                            $csv_content = substr_replace($csv_content, "", -1);
                        }
                    }
                }

                $student_id_count = $attendance_of_student['student_id'];
            }
        }

        $txt = fopen($file, "w") or die("Unable to open file!");
        fwrite($txt, $csv_content);
        fclose($txt);

        header('Content-Description: File Transfer');
        header('Content-Disposition: attachment; filename=' . $file);
        header('Expires: 0');
        header('Cache-Control: must-revalidate');
        header('Pragma: public');
        header('Content-Length: ' . filesize($file));
        header("Content-type: text/csv");
        readfile($file);
    }

    /**
     * Show the routine.
     *
     * @return \Illuminate\Contracts\Support\Renderable
     */
    public function routine()
    {
        $student_data = (new CommonController)->get_student_details_by_id(auth()->user()->id);
        $class_id     = $student_data['class_id'];
        $section_id   = $student_data['section_id'];
        $classes      = Classes::where('school_id', auth()->user()->school_id)->get();
        return view('student.routine.routine', ['class_id' => $class_id, 'section_id' => $section_id, 'classes' => $classes]);
    }

    /**
     * Show the subject list.
     *
     * @return \Illuminate\Contracts\Support\Renderable
     */
    public function subjectList()
    {
        $active_session = get_school_settings(auth()->user()->school_id)->value('running_session');

        $student_data = (new CommonController)->get_student_details_by_id(auth()->user()->id);
        $subjects     = Subject::where('class_id', $student_data['class_id'])
            ->where('school_id', auth()->user()->school_id)
            ->where('session_id', $active_session)
            ->paginate(10);

        return view('student.subject.subject_list', compact('subjects'));
    }

    /**
     * Show the syllabus.
     *
     * @return \Illuminate\Contracts\Support\Renderable
     */
    public function syllabus()
    {
        if (auth()->user()->role_id != "" && auth()->user()->role_id == 7) {
            $active_session = get_school_settings(auth()->user()->school_id)->value('running_session');
            $student_data   = (new CommonController)->get_student_details_by_id(auth()->user()->id);

            $syllabuses = Syllabus::where(['class_id' => $student_data['class_id'], 'section_id' => $student_data['section_id'], 'session_id' => $active_session, 'school_id' => auth()->user()->school_id])->paginate(10);

            return view('student.syllabus.syllabus', compact('syllabuses'));
        } else {
            return redirect('login')->with('error', "Please login first.");
        }
    }

    /**
     * Show the grade list.
     *
     * @return \Illuminate\Contracts\Support\Renderable
     */
    public function marks($value = '')
    {
        $exam_categories = ExamCategory::where('school_id', auth()->user()->school_id)->get();
        $user_id         = auth()->user()->id;
        $student_details = (new CommonController)->get_student_details_by_id($user_id);

        $subjects = Subject::where(['class_id' => $student_details['class_id'], 'school_id' => auth()->user()->school_id])->get();

        return view('student.marks.index', ['exam_categories' => $exam_categories, 'student_details' => $student_details, 'subjects' => $subjects]);
    }

    public function gradeList()
    {
        $grades = Grade::where('school_id', auth()->user()->school_id)->paginate(10);
        return view('student.grade.grade_list', compact('grades'));
    }

    /**
     * Show the book list.
     *
     * @return \Illuminate\Contracts\Support\Renderable
     */
    public function bookList(Request $request)
    {
        $search = $request['search'] ?? "";

        if ($search != "") {

            $books = Book::where(function ($query) use ($search) {
                $query->where('name', 'LIKE', "%{$search}%")
                    ->where('school_id', auth()->user()->school_id);
            })->orWhere(function ($query) use ($search) {
                $query->where('author', 'LIKE', "%{$search}%")
                    ->where('school_id', auth()->user()->school_id);
            })->paginate(10);
        } else {
            $books = Book::where('school_id', auth()->user()->school_id)->paginate(10);
        }

        return view('student.book.list', compact('books', 'search'));
    }

    public function bookIssueList(Request $request)
    {
        $active_session = get_school_settings(auth()->user()->school_id)->value('running_session');

        if (count($request->all()) > 0) {

            $data = $request->all();

            $date        = explode('-', $data['eDateRange']);
            $date_from   = strtotime($date[0] . ' 00:00:00');
            $date_to     = strtotime($date[1] . ' 23:59:59');
            $book_issues = BookIssue::where('issue_date', '>=', $date_from)
                ->where('issue_date', '<=', $date_to)
                ->where('school_id', auth()->user()->school_id)
                ->where('session_id', $active_session)
                ->get();

            return view('student.book.book_issue', ['book_issues' => $book_issues, 'date_from' => $date_from, 'date_to' => $date_to]);
        } else {

            $date_from   = strtotime(date('d-M-Y', strtotime(' -30 day')) . ' 00:00:00');
            $date_to     = strtotime(date('d-M-Y') . ' 23:59:59');
            $book_issues = BookIssue::where('issue_date', '>=', $date_from)
                ->where('issue_date', '<=', $date_to)
                ->where('school_id', auth()->user()->school_id)
                ->where('session_id', $active_session)
                ->get();

            return view('student.book.book_issue', ['book_issues' => $book_issues, 'date_from' => $date_from, 'date_to' => $date_to]);
        }
    }

    /**
     * Show the noticeboard list.
     *
     * @return \Illuminate\Contracts\Support\Renderable
     */
    public function noticeboardList()
    {
        $notices = Noticeboard::get()->where('school_id', auth()->user()->school_id);

        $events = [];

        foreach ($notices as $notice) {
            if ($notice['end_date'] != "") {
                if ($notice['start_date'] != $notice['end_date']) {
                    $end_date = strtotime($notice['end_date']) + 24 * 60 * 60;
                    $end_date = date('Y-m-d', $end_date);
                } else {
                    $end_date = date('Y-m-d', strtotime($notice['end_date']));
                }
            }

            if ($notice['end_date'] == "" && $notice['start_time'] == "" && $notice['end_time'] == "") {
                $info = [
                    'id'    => $notice['id'],
                    'title' => $notice['notice_title'],
                    'start' => date('Y-m-d', strtotime($notice['start_date'])),
                ];
            } else if ($notice['start_time'] != "" && ($notice['end_date'] == "" && $notice['end_time'] == "")) {
                $info = [
                    'id'    => $notice['id'],
                    'title' => $notice['notice_title'],
                    'start' => date('Y-m-d', strtotime($notice['start_date'])) . 'T' . $notice['start_time'],
                ];
            } else if ($notice['end_date'] != "" && ($notice['start_time'] == "" && $notice['end_time'] == "")) {
                $info = [
                    'id'    => $notice['id'],
                    'title' => $notice['notice_title'],
                    'start' => date('Y-m-d', strtotime($notice['start_date'])),
                    'end'   => $end_date,
                ];
            } else if ($notice['end_date'] != "" && $notice['start_time'] != "" && $notice['end_time'] != "") {
                $info = [
                    'id'    => $notice['id'],
                    'title' => $notice['notice_title'],
                    'start' => date('Y-m-d', strtotime($notice['start_date'])) . 'T' . $notice['start_time'],
                    'end'   => date('Y-m-d', strtotime($notice['end_date'])) . 'T' . $notice['end_time'],
                ];
            } else {
                $info = [
                    'id'    => $notice['id'],
                    'title' => $notice['notice_title'],
                    'start' => date('Y-m-d', strtotime($notice['start_date'])),
                ];
            }
            array_push($events, $info);
        }

        $events = json_encode($events);

        return view('student.noticeboard.noticeboard', ['events' => $events]);
    }

    public function editNoticeboard($id = "")
    {
        $notice = Noticeboard::find($id);
        return view('student.noticeboard.edit', ['notice' => $notice]);
    }

    /**
     * Show the live class.
     *
     * @return \Illuminate\Contracts\Support\Renderable
     */

    public function FeeManagerList(Request $request)
    {
        $active_session            = get_school_settings(auth()->user()->school_id)->value('running_session');
        $student_class_information = Enrollment::where('user_id', auth()->user()->id)->first()->toArray();

        if (count($request->all()) > 0) {

            $data            = $request->all();
            $date            = explode('-', $data['eDateRange']);
            $date_from       = strtotime($date[0] . ' 00:00:00');
            $date_to         = strtotime($date[1] . ' 23:59:59');
            $selected_status = $data['status'];

            if ($selected_status != "all") {
                $invoices = StudentFeeManager::where('timestamp', '>=', $date_from)->where('timestamp', '<=', $date_to)->where('status', $selected_status)->where('student_id', auth()->user()->id)->where('session_id', $active_session)->get();
            } else if ($selected_status == "all") {
                $invoices = StudentFeeManager::where('timestamp', '>=', $date_from)->where('timestamp', '<=', $date_to)->where('school_id', auth()->user()->school_id)->where('student_id', auth()->user()->id)->where('session_id', $active_session)->get();
            }

            return view('student.fee_manager.student_fee_manager', ['invoices' => $invoices, 'date_from' => $date_from, 'date_to' => $date_to, 'selected_status' => $selected_status]);
        } else {

            $date_from       = strtotime(date('d-M-Y', strtotime(' -30 day')) . ' 00:00:00');
            $date_to         = strtotime(date('d-M-Y') . ' 23:59:59');
            $selected_status = "";

            $invoices = StudentFeeManager::where('timestamp', '>=', $date_from)->where('timestamp', '<=', $date_to)->where('student_id', auth()->user()->id)->where('school_id', auth()->user()->school_id)->where('session_id', $active_session)->get();

            return view('student.fee_manager.student_fee_manager', ['invoices' => $invoices, 'date_from' => $date_from, 'date_to' => $date_to, 'selected_status' => $selected_status]);
        }
    }

    public function feeManagerExport($date_from = "", $date_to = "", $selected_status = "")
    {

        $active_session = get_school_settings(auth()->user()->school_id)->value('running_session');

        if ($selected_status != "all") {
            $invoices = StudentFeeManager::where('timestamp', '>=', $date_from)->where('timestamp', '<=', $date_to)->where('status', $selected_status)->where('student_id', auth()->user()->id)->where('session_id', $active_session)->get();
        } else if ($selected_status == "all") {
            $invoices = StudentFeeManager::where('timestamp', '>=', $date_from)->where('timestamp', '<=', $date_to)->where('school_id', auth()->user()->school_id)->where('student_id', auth()->user()->id)->where('session_id', $active_session)->get();
        }

        $classes = Classes::where('school_id', auth()->user()->school_id)->get();

        $file = "student_fee-" . date('d-m-Y', $date_from) . '-' . date('d-m-Y', $date_to) . '-' . $selected_status . ".csv";

        $csv_content = get_phrase('Invoice No') . ', ' . get_phrase('Student') . ', ' . get_phrase('Class') . ', ' . get_phrase('Invoice Title') . ', ' . get_phrase('Total Amount') . ', ' . get_phrase('Created At') . ', ' . get_phrase('Paid Amount') . ', ' . get_phrase('Status');

        foreach ($invoices as $invoice) {
            $csv_content .= "\n";

            $student_details = (new CommonController)->get_student_details_by_id($invoice['student_id']);
            $invoice_no      = sprintf('%08d', $invoice['id']);

            $csv_content .= $invoice_no . ', ' . $student_details['name'] . ', ' . $student_details['class_name'] . ', ' . $invoice['title'] . ', ' . currency($invoice['total_amount']) . ', ' . date('d-M-Y', $invoice['timestamp']) . ', ' . currency($invoice['paid_amount']) . ', ' . $invoice['status'];
        }
        $txt = fopen($file, "w") or die("Unable to open file!");
        fwrite($txt, $csv_content);
        fclose($txt);

        header('Content-Description: File Transfer');
        header('Content-Disposition: attachment; filename=' . $file);
        header('Expires: 0');
        header('Cache-Control: must-revalidate');
        header('Pragma: public');
        header('Content-Length: ' . filesize($file));
        header("Content-type: text/csv");
        readfile($file);
    }

    public function FeePayment(Request $request, $id)
    {

        $fee_details = StudentFeeManager::where('id', $id)->first()->toArray();
        $user_info   = User::where('id', $fee_details['student_id'])->first()->toArray();
        return view('student.payment.payment_gateway', ['fee_details' => $fee_details, 'user_info' => $user_info]);
    }

    public function studentFeeinvoice(Request $request, $id)
    {

        $invoice_details = StudentFeeManager::find($id)->toArray();
        $student_details = (new CommonController)->get_student_details_by_id($invoice_details['student_id'])->toArray();

        return view('student.fee_manager.invoice', ['invoice_details' => $invoice_details, 'student_details' => $student_details]);
    }

    public function offlinePaymentStudent(Request $request, $id = "")
    {
        $data = $request->all();

        if ($data['amount'] > 0) {

            $file = $data['document_image'];

            if ($file) {
                $filename  = $file->getClientOriginalName();
                $extension = $file->getClientOriginalExtension(); //Get extension of uploaded file

                $file->move(public_path('assets/uploads/offline_payment'), $filename);
                $data['document_image'] = $filename;
            } else {
                $data['document_image'] = '';
            }

            StudentFeeManager::where('id', $id)->update([
                'status'         => 'pending',
                'document_image' => $data['document_image'],
                'payment_method' => 'offline',
            ]);

            return redirect()->route('student.fee_manager.list')->with('message', 'offline payment requested successfully');
        } else {
            return redirect()->route('student.fee_manager.list')->with('message', 'offline payment requested fail');
        }
    }

    public function profile()
    {
        return view('student.profile.view');
    }

    public function profile_update(Request $request)
    {
        $data['name']  = $request->name;
        $data['email'] = $request->email;

        $user_info['birthday'] = strtotime($request->eDefaultDateRange);
        $user_info['gender']   = $request->gender;
        $user_info['phone']    = $request->phone;
        $user_info['address']  = $request->address;

        if (empty($request->photo)) {
            $user_info['photo'] = $request->old_photo;
        } else {
            $file_name          = random(10) . '.png';
            $user_info['photo'] = $file_name;

            $request->photo->move(public_path('assets/uploads/user-images/'), $file_name);
        }

        $data['user_information'] = json_encode($user_info);

        User::where('id', auth()->user()->id)->update($data);

        return redirect(route('student.profile'))->with('message', get_phrase('Profile info updated successfully'));
    }

    public function user_language(Request $request)
    {
        $data['language'] = $request->language;
        User::where('id', auth()->user()->id)->update($data);

        return redirect()->back()->with('message', 'You have successfully transleted language.');
    }

    public function password($action_type = null, Request $request)
    {

        if ($action_type == 'update') {

            if ($request->new_password != $request->confirm_password) {
                return back()->with("error", "Confirm Password Doesn't match!");
            }

            if (! Hash::check($request->old_password, auth()->user()->password)) {
                return back()->with("error", "Current Password Doesn't match!");
            }

            $data['password'] = Hash::make($request->new_password);
            User::where('id', auth()->user()->id)->update($data);

            return redirect(route('student.password', 'edit'))->with('message', get_phrase('Password changed successfully'));
        }

        return view('student.profile.password');
    }

    /**
     * Show the event list.
     *
     * @return \Illuminate\Contracts\Support\Renderable
     */
    public function eventList(Request $request)
    {
        $search = $request['search'] ?? "";

        if ($search != "") {

            $events = FrontendEvent::where(function ($query) use ($search) {
                $query->where('title', 'LIKE', "%{$search}%");
            })->paginate(10);
        } else {
            $events = FrontendEvent::where('school_id', auth()->user()->school_id)->paginate(10);
        }

        return view('student.events.events', compact('events', 'search'));
    }

    public function complain()
    {
        return view('student.complain.complain');
    }
    public function complainUser(Request $request)
    {
        $data = $request->all();

        $page_data['class_id']   = $data['class_id'];
        $page_data['section_id'] = $data['section_id'];
        $page_data['receiver']   = $data['receiver'];
        return view('student.complain.complainUser', ['page_data' => $page_data]);
    }

    public function receivers(Request $request)
    {
        $data = $request->all();

        $page_data['class_id']   = $data['class_id'];
        $page_data['section_id'] = $data['section_id'];
        $page_data['receiver']   = $data['receiver'];
        return view('student.complain.complain', ['page_data' => $page_data]);
    }

    //  Message

    public function allMessage(Request $request, $id)
    {

        $msg_user_details = DB::table('users')
            ->join('message_thrades', function ($join) {
                // Join where the user is the sender
                $join->on('users.id', '=', 'message_thrades.sender_id')
                    ->orWhere(function ($query) {
                        // Join where the user is the receiver
                        $query->on('users.id', '=', 'message_thrades.reciver_id');
                    });
            })
            ->select('users.id as user_id', 'message_thrades.id as thread_id', 'users.*', 'message_thrades.*')
            ->where('message_thrades.id', $id)
            ->where('message_thrades.school_id', auth()->user()->school_id)
            ->where('users.id', '<>', auth()->user()->id) // Exclude the authenticated user
            ->first();

        if ($request->ajax()) {
            $query = $request->input('query');

            // Search users by name or any other criteria
            $users = User::where('name', 'LIKE', "%{$query}%")
                ->where('school_id', auth()->user()->school_id)
                ->get();

            // Prepare HTML response
            $html = '';

            // Check if any users were found
            if ($users->isEmpty()) {
                return response()->json('No User found');
            }

            foreach ($users as $user) {

                if (! empty($user)) {
                    $userInfo = json_decode($user->user_information);

                    $user_image = ! empty($userInfo->photo)
                        ? asset('assets/uploads/user-images/' . $userInfo->photo)
                        : asset('assets/uploads/user-images/thumbnail.png');

                    $html .= '
                          <div class="user-item d-flex align-items-center msg_us_src_list">
                              <a href="' . route('student.message.messagethrades', ['id' => $user->id]) . '">
                                  <img src="' . $user_image . '" alt="User Image" style="width: 50px; height: 50px; border-radius: 50%;">
                                  <span class="ms-3">' . $user->name . '</span>
                              </a>
                          </div>
                      ';
                }
            }

            return response()->json($html);
        }

        $chat_datas = Chat::where('school_id', auth()->user()->school_id)->get();

        $counter_condition = Chat::where('message_thrade', $id)->orderBy('id', 'desc')->first();

        if ($counter_condition->sender_id != auth()->user()->id) {
            Chat::where('message_thrade', $id)->update(['read_status' => 1]);
        }

        return view('student.message.all_message', ['msg_user_details' => $msg_user_details], ['chat_datas' => $chat_datas]);
    }

    public function messagethrades($id)
    {

        $exists = MessageThrade::where('reciver_id', $id)
            ->where('sender_id', auth()->user()->id)
            ->exists();
        if ($id != auth()->user()->id) {
            if (! $exists) {
                $message_thrades_data = [
                    'reciver_id' => $id,
                    'sender_id'  => auth()->user()->id,
                    'school_id'  => auth()->user()->school_id,
                ];

                MessageThrade::create($message_thrades_data);

                //return redirect()->back()->with('message', 'User added successfully');
            }

            $message_thrades = MessageThrade::where('reciver_id', $id)
                ->where('sender_id', auth()->user()->id)
                ->first();
            $msg_trd_id = $message_thrades->id;

            $msg_user_details = DB::table('users')
                ->join('message_thrades', 'users.id', '=', 'message_thrades.reciver_id')
                ->select('users.id as user_id', 'message_thrades.id as thread_id', 'users.*', 'message_thrades.*')
                ->where('message_thrades.id', $msg_trd_id)
                ->first();

            $chat_datas = Chat::where('school_id', auth()->user()->school_id)->get();

            // Combine all data into a single array
            return view('student.message.all_message', ['id' => $msg_trd_id, 'msg_user_details' => $msg_user_details, 'chat_datas' => $chat_datas]);
        }
        return redirect()->back()->with('error', 'You can not add you');
    }

    public function chat_save(Request $request)
    {
        $data      = $request->all();
        $chat_data = [
            'message_thrade' => $data['message_thrade'],
            'reciver_id'     => $data['reciver_id'],
            'message'        => $data['message'],
            'school_id'      => auth()->user()->school_id,
            'sender_id'      => auth()->user()->id,
            'read_status'    => 0,

        ];

        // Create feedback entry
        Chat::create($chat_data);

        return redirect()->back();
    }

    public function chat_empty(Request $request)
    {

        if ($request->ajax()) {
            $query = $request->input('query');

            $users = User::where('name', 'LIKE', "%{$query}%")
                ->where('school_id', auth()->user()->school_id)
                ->get();

            $html = '';

            if ($users->isEmpty()) {
                return response()->json('No User found');
            }

            foreach ($users as $user) {
                $userInfo   = json_decode($user->user_information);
                $user_image = ! empty($userInfo->photo)
                    ? asset('assets/uploads/user-images/' . $userInfo->photo)
                    : asset('assets/uploads/user-images/thumbnail.png');

                $html .= '
                      <div class="user-item d-flex align-items-center msg_us_src_list">
                          <a href="' . route('student.message.messagethrades', ['id' => $user->id]) . '">
                              <img src="' . $user_image . '" alt="User Image" style="width: 50px; height: 50px; border-radius: 50%;">
                              <span class="ms-3">' . $user->name . '</span>
                          </a>
                      </div>
                  ';
            }

            return response()->json($html);
        }

        // Pass the data to the view only if msg_user_details is not null
        return view('student.message.chat_empty');
    }

    public function appraisalList()
    {
        $user_info = auth()->user();

        $student_details = (new CommonController)->get_student_details_by_id($user_info->id);

        $appraisals = Appraisal::where('class_id', $student_details->class_id)->where('school_id', auth()->user()->school_id)->where('status', 1)->paginate(10);

        return view('student.appraisal.appraisalList', ['appraisals' => $appraisals]);
    }

    public function singleAppraisal($id)
    {
        $student_id = auth()->user()->id;

        $appraisal = Appraisal::where('id', $id)->first();
        // Check if the student has already submitted
        $submission = Appraisal_submit::where([
            ['apprasial_id', $id],
            ['student_id', $student_id],
        ])->first();

        $submittedAnswers = $submission ? json_decode($submission->answers, true) : [];
        return view('student.appraisal.singleAppraisal', compact('appraisal', 'submittedAnswers'));
    }

    public function appraisalSubmit(Request $request, $id)
    {
        $student_id = auth()->user()->id;

        // Check if already submitted
        if (Appraisal_submit::where('apprasial_id', $id)->where('student_id', $student_id)->exists()) {
            return redirect()->back()->with('error', 'You have already submitted this appraisal.');
        }

        $appraisal  = Appraisal::findOrFail($id);
        $student_id = auth()->user()->id;
        $school_id  = auth()->user()->school_id;
        $answers    = $request->input('answers');

        Appraisal_submit::create([
            'apprasial_id' => $id,
            'student_id'   => $student_id,
            'school_id'    => $school_id,
            'answers'      => json_encode($answers),
        ]);

        return redirect()->back()->with('message', 'Appraisal submitted successfully!');
    }

    public function hostelApplications()
    {
        $student_id = auth()->user()->id;

        $applications = HostelApplication::where('student_id', $student_id)->where('school_id', auth()->user()->school_id)->paginate(10);

        return view('student.hostel.applications.list', compact('applications'));
    }
    public function applicationCreate()
    {
        $hostels = Hostel::get();

        $hostel_rooms = HostelRoom::where('status', 1)
            ->whereRaw('occupied < capacity')
            ->with('hostel')
            ->get();

        return view('student.hostel.applications.create', compact('hostels', 'hostel_rooms'));
    }
    public function applicationStore(Request $request)
    {
        $request->validate([
            'hostel_id' => 'required|exists:hostels,id',
            'room_id'   => 'required|exists:hostel_rooms,id',
            'note'      => 'nullable|string|max:500',
        ]);

        $existingApplication = HostelApplication::where('student_id', auth()->user()->id)
            ->whereIn('status', [0, 1])
            ->first();

        if ($existingApplication) {
            return redirect()->back()->with('error', get_phrase('You already have an active hostel application'));
        }

        $existingAllocation = HostelRoomAllocation::where('student_id', auth()->user()->id)
            ->where('status', 1)
            ->first();

        if ($existingAllocation) {
            return redirect()->back()->with('error', get_phrase('You are already allocated to a hostel room'));
        }

        $room = HostelRoom::find($request->room_id);
        if ($room->occupied >= $room->capacity) {
            return redirect()->back()->with('error', get_phrase('Selected room is already full'));
        }

        if ($room->hostel_id != $request->hostel_id) {
            return redirect()->back()->with('error', get_phrase('Selected room does not belong to the chosen hostel'));
        }

        $application             = new HostelApplication();
        $application->student_id = auth()->user()->id;
        $application->school_id  = auth()->user()->school_id;
        $application->hostel_id  = $request->hostel_id;
        $application->room_id    = $request->room_id;
        $application->status     = 0;
        $application->note       = $request->note;
        $application->save();

        return redirect()->route('student.hostel.applications')->with('success', get_phrase('Hostel application submitted successfully'));
    }
    /**
     * Show the hostel fee manager list.
     *
     * @return \Illuminate\Contracts\Support\Renderable
     */
    public function hostelFeeManagerList(Request $request)
    {
        $active_session = get_school_settings(auth()->user()->school_id)->value('running_session');
        $student_id     = auth()->user()->id;

        if (count($request->all()) > 0) {
            $data            = $request->all();
            $date            = explode('-', $data['eDateRange']);
            $date_from       = strtotime($date[0] . ' 00:00:00');
            $date_to         = strtotime($date[1] . ' 23:59:59');
            $selected_status = $data['status'];

            if ($selected_status != "all") {
                $invoices = HostelFee::where('created_at', '>=', date('Y-m-d H:i:s', $date_from))
                    ->where('created_at', '<=', date('Y-m-d H:i:s', $date_to))
                    ->where('status', $selected_status)
                    ->where('student_id', $student_id)
                    ->where('school_id', auth()->user()->school_id)
                    ->get();
            } else {
                $invoices = HostelFee::where('created_at', '>=', date('Y-m-d H:i:s', $date_from))
                    ->where('created_at', '<=', date('Y-m-d H:i:s', $date_to))
                    ->where('student_id', $student_id)
                    ->where('school_id', auth()->user()->school_id)
                    ->get();
            }

            return view('student.hostel.fee_manager.student_hostel_fee_manager', [
                'invoices'        => $invoices,
                'date_from'       => $date_from,
                'date_to'         => $date_to,
                'selected_status' => $selected_status,
            ]);
        } else {
            $date_from       = strtotime(date('d-M-Y', strtotime(' -30 day')) . ' 00:00:00');
            $date_to         = strtotime(date('d-M-Y') . ' 23:59:59');
            $selected_status = "";

            $invoices = HostelFee::where('created_at', '>=', date('Y-m-d H:i:s', $date_from))
                ->where('created_at', '<=', date('Y-m-d H:i:s', $date_to))
                ->where('student_id', $student_id)
                ->where('school_id', auth()->user()->school_id)
                ->get();

            return view('student.hostel.fee_manager.student_hostel_fee_manager', [
                'invoices'        => $invoices,
                'date_from'       => $date_from,
                'date_to'         => $date_to,
                'selected_status' => $selected_status,
            ]);
        }
    }

    public function hostelFeeManagerExport($date_from = "", $date_to = "", $selected_status = "")
    {
        $student_id = auth()->user()->id;

        if ($selected_status != "all") {
            $invoices = HostelFee::where('created_at', '>=', date('Y-m-d H:i:s', $date_from))
                ->where('created_at', '<=', date('Y-m-d H:i:s', $date_to))
                ->where('status', $selected_status)
                ->where('student_id', $student_id)
                ->where('school_id', auth()->user()->school_id)
                ->get();
        } else {
            $invoices = HostelFee::where('created_at', '>=', date('Y-m-d H:i:s', $date_from))
                ->where('created_at', '<=', date('Y-m-d H:i:s', $date_to))
                ->where('student_id', $student_id)
                ->where('school_id', auth()->user()->school_id)
                ->get();
        }

        $file        = "hostel_fee-" . date('d-m-Y', $date_from) . '-' . date('d-m-Y', $date_to) . '-' . $selected_status . ".csv";
        $csv_content = get_phrase('Invoice No') . ', ' . get_phrase('Student') . ', ' . get_phrase('Hostel') . ', ' . get_phrase('Invoice Title') . ', ' . get_phrase('Total Amount') . ', ' . get_phrase('Created At') . ', ' . get_phrase('Paid Amount') . ', ' . get_phrase('Status');

        foreach ($invoices as $invoice) {
            $csv_content .= "\n";
            $student_details = (new CommonController)->get_student_details_by_id($invoice['student_id']);
            $hostel          = Hostel::find($invoice['hostel_id']);
            $invoice_no      = sprintf('%08d', $invoice['id']);

            $csv_content .= $invoice_no . ', ' . $student_details['name'] . ', ' . ($hostel ? $hostel->name : 'N/A') . ', ' . $invoice['title'] . ', ' . currency($invoice['amount']) . ', ' . date('d-M-Y', strtotime($invoice['created_at'])) . ', ' . currency($invoice['paid_amount']) . ', ' . $invoice['status'];
        }

        $txt = fopen($file, "w") or die("Unable to open file!");
        fwrite($txt, $csv_content);
        fclose($txt);

        header('Content-Description: File Transfer');
        header('Content-Disposition: attachment; filename=' . $file);
        header('Expires: 0');
        header('Cache-Control: must-revalidate');
        header('Pragma: public');
        header('Content-Length: ' . filesize($file));
        header("Content-type: text/csv");
        readfile($file);
    }

    /**
     * Show hostel fee invoice.
     *
     * @return \Illuminate\Contracts\Support\Renderable
     */
    public function hostelFeeInvoice(Request $request, $id)
    {
        $invoice_details = HostelFee::find($id)->toArray();
        $student_details = (new CommonController)->get_student_details_by_id($invoice_details['student_id'])->toArray();
        $hostel          = Hostel::find($invoice_details['hostel_id']);

        return view('student.hostel.fee_manager.invoice', [
            'invoice_details' => $invoice_details,
            'student_details' => $student_details,
            'hostel'          => $hostel,
        ]);
    }

    public function hostelFeePayment(Request $request)
    {
        $validated = $request->validate([
            'month' => 'required|integer|min:1|max:12',
            'year'  => 'required|integer',
        ]);

        $application = HostelApplication::where('student_id', auth()->user()->id)
            ->where('school_id', auth()->user()->school_id)
            ->first();

        $fee_amount = HostelRoom::where('id', $application->room_id)
            ->where('school_id', auth()->user()->school_id)
            ->value('seat_fee');

        $hostelFee = HostelFee::create([
            'student_id'       => auth()->user()->id,
            'school_id'        => auth()->user()->school_id,
            'hostel_id'        => $application->hostel_id,
            'title'            => get_phrase('Hostel Fee Payment'),
            'amount'           => $fee_amount,
            'status'           => 0,
            'fee_payment_date' => \Carbon\Carbon::createFromDate($validated['year'], $validated['month'], 1)->format('Y-m-d'),
        ]);

        $fee_details = [
            'id'               => $hostelFee->id,
            'title'            => 'hostel_fee',
            'total_amount'     => $fee_amount,
            'class_id'         => null,
            'parent_id'        => null,
            'student_id'       => auth()->user()->id,
            'payment_method'   => null,
            'paid_amount'      => 0,
            'status'           => 'unpaid',
            'school_id'        => auth()->user()->school_id,
            'session_id'       => get_school_settings(auth()->user()->school_id)->value('running_session'),
            'timestamp'        => time(),
            'discounted_price' => 0,
            'amount'           => $fee_amount,
        ];
        $user_info = User::where('id', auth()->user()->id)->first()->toArray();

        session()->put([
            'hostel_fee_month' => $validated['month'],
            'hostel_fee_year'  => $validated['year'],

        ]);
        return view('student.hostel.payment_gateway', compact('fee_details', 'user_info'));
    }

    public function offlinePaymentHostel(Request $request)
    {
        $data = $request->all();

        $application = HostelApplication::where('student_id', auth()->user()->id)
            ->where('school_id', auth()->user()->school_id)
            ->firstOrFail();

        if ($data['amount'] <= 0) {
            return redirect()->route('student.hostel_fee_manager.list')
                ->with('message', 'Offline payment request failed');
        }

        $fileName = '';
        if ($request->hasFile('document_image')) {
            $file     = $request->file('document_image');
            $fileName = preg_replace('/[^a-zA-Z0-9._-]/', '', $file->getClientOriginalName());
            $file->move(public_path('assets/uploads/hostel_fees'), $fileName);
        }

        $feePaymentDate = Carbon::createFromDate(
            session('hostel_fee_year'),
            session('hostel_fee_month'),
            1
        )->format('Y-m-d');

        // Check if a rejected payment already exists for this month/year
        $existingFee = HostelFee::where('student_id', auth()->user()->id)
            ->where('school_id', auth()->user()->school_id)
            ->where('fee_payment_date', $feePaymentDate)
            ->whereIn('status', [0, 2])
            ->first();

        if ($existingFee) {
            // Update the rejected or pending payment
            $existingFee->amount         = $data['amount'];
            $existingFee->document_image = $fileName;
            $existingFee->status         = 0;
            $existingFee->save();
        } else {
            // Create new payment record
            HostelFee::create([
                'school_id'        => auth()->user()->school_id,
                'student_id'       => auth()->user()->id,
                'hostel_id'        => $application->hostel_id,
                'title'            => get_phrase('Hostel Fee Payment'),
                'amount'           => $data['amount'],
                'status'           => 0,
                'fee_payment_date' => $feePaymentDate,
                'document_image'   => $fileName,
            ]);
        }

        // Clear session data
        session()->forget(['payment_for', 'fee_details', 'hostel_fee_month', 'hostel_fee_year']);

        // Redirect to hostel fee manager list with success message
        return redirect()->route('student.hostel_fee_manager.list')->with('message', 'Offline payment requested successfully');
    }

    /**
     * Handle successful hostel fee payment.
     *
     * @return \Illuminate\Http\RedirectResponse
     */
    public function student_hostel_fee_success_payment_student($user_data = "", $response = "")
    {
        $response  = json_decode($response, true);
        $user_data = json_decode($user_data, true);

        HostelFee::where('id', $user_data['invoice_id'])->update([
            'paid_amount'    => $user_data['total_amount'],
            'status'         => 'paid',
            'payment_method' => $user_data['payment_method'],
            'payment_date'   => date('Y-m-d H:i:s'),
        ]);

        return redirect()->route('student.hostel_fee_manager.list')->with('message', 'Payment completed successfully');
    }

    /**
     * Handle failed hostel fee payment.
     *
     * @return \Illuminate\Http\RedirectResponse
     */
    public function student_hostel_fee_fail_payment_student($user_data = "", $response = "")
    {
        return redirect()->route('student.hostel_fee_manager.list')->with('error', 'Payment failed. Please try again.');
    }

    public function hostelFeeMonthlyList()
    {
        $student_id = auth()->user()->id;

        // Check if student is allocated to any hostel
        $allocation = HostelRoomAllocation::where('student_id', $student_id)
            ->where('status', 1)
            ->first();

        if (! $allocation) {
            return redirect()->back()->with('error', 'You are not allocated to any hostel.');
        }

        $currentYear = now()->year;
        $monthlyFees = [];

        // Get all months from January to current month
        for ($month = 1; $month <= now()->month; $month++) {
            $monthName  = date('F', mktime(0, 0, 0, $month, 1));
            $monthStart = \Carbon\Carbon::create($currentYear, $month, 1)->startOfDay();
            $monthEnd   = \Carbon\Carbon::create($currentYear, $month, 1)->endOfMonth();

            $feePaid = HostelFee::where('school_id', auth()->user()->school_id)
                ->where('student_id', $student_id)
                ->whereBetween('created_at', [$monthStart, $monthEnd])
                ->first();

            $monthlyFees[] = [
                'month'              => $monthName,
                'year'               => $currentYear,
                'status'             => $feePaid ? 'Paid' : 'Unpaid',
                'fee_details'        => $feePaid,
                'month_number'       => $month,
                'is_current_or_past' => $month <= now()->month,
            ];
        }

        return view('student.hostel.fee_manager.monthly_list', [
            'monthlyFees' => $monthlyFees,
            'allocation'  => $allocation,
            'currentYear' => $currentYear,
        ]);
    }

    /**
     * Pay specific month hostel fee.
     *
     * @return \Illuminate\Contracts\Support\Renderable
     */
    public function payMonthlyFee(Request $request, $month, $year)
    {
        $student_id = auth()->user()->id;

        // Check if student is allocated to any hostel
        $allocation = HostelRoomAllocation::where('student_id', $student_id)
            ->where('status', 1)
            ->first();

        if (! $allocation) {
            return redirect()->back()->with('error', 'You are not allocated to any hostel.');
        }

        // Check if fee already paid for this month
        $monthStart = \Carbon\Carbon::create($year, $month, 1)->startOfDay();
        $monthEnd   = \Carbon\Carbon::create($year, $month, 1)->endOfMonth();

        $existingFee = HostelFee::where('school_id', auth()->user()->school_id)
            ->where('student_id', $student_id)
            ->whereBetween('created_at', [$monthStart, $monthEnd])
            ->first();

        if ($existingFee) {
            return redirect()->route('student.hostel_fee.monthly_list')->with('error', 'Fee for ' . date('F Y', mktime(0, 0, 0, $month, 1, $year)) . ' already paid.');
        }

        // Get hostel details
        $hostel = Hostel::find($allocation->hostel_id);
        $room   = HostelRoom::find($allocation->room_id);

        // Create fee record
        $fee                   = new HostelFee();
        $fee->student_id       = $student_id;
        $fee->school_id        = auth()->user()->school_id;
        $fee->hostel_id        = $allocation->hostel_id;
        $fee->room_id          = $allocation->room_id;
        $fee->title            = 'Hostel Fee for ' . date('F Y', mktime(0, 0, 0, $month, 1, $year));
        $fee->amount           = $hostel->fee ?? 0;
        $fee->fee_payment_date = \Carbon\Carbon::createFromDate($year, $month, 1)->format('Y-m-d');
        $fee->status           = 0;
        $fee->due_date         = $monthEnd;
        $fee->save();

        $fee_details = [
            'id'               => $fee->id,
            'title'            => $fee->title,
            'total_amount'     => $fee->amount,
            'class_id'         => null,
            'parent_id'        => null,
            'student_id'       => $student_id,
            'payment_method'   => null,
            'paid_amount'      => 0,
            'status'           => 'unpaid',
            'school_id'        => auth()->user()->school_id,
            'session_id'       => get_school_settings(auth()->user()->school_id)->value('running_session'),
            'timestamp'        => time(),
            'discounted_price' => 0,
            'amount'           => $fee->amount,
        ];
        $user_info = User::where('id', $student_id)->first()->toArray();
        return view('student.payment.payment_gateway', compact('fee_details', 'user_info'));
    }

    public function club(Request $request)
    {
        $search     = $request->search;
        $advisorId = $request->advisor_id;

        $clubs = Club::with('advisor')
            ->when($search, function ($query) use ($search) {
                $query->where('club_name', 'LIKE', "%{$search}%");
            })
            ->when($advisorId, function ($query) use ($advisorId) {
                $query->where('advisor_id', $advisorId);
            })
            ->latest()
            ->paginate(10)
            ->withQueryString();

        $teachers = User::where('role_id', 3)
            ->where('status', 1)
            ->get();

        return view('student.club.index', compact(
            'clubs',
            'search',
            'advisorId',
            'teachers'
        ));
    }


    public function join(Club $club)
    {
        $studentId = auth()->id();

        $exists = ClubMember::where('club_id', $club->id)
            ->where('student_id', $studentId)
            ->first();

        if ($exists) {
            return back()->with('error', 'Already requested');
        }

        ClubMember::create([
            'club_id' => $club->id,
            'student_id' => $studentId,
            'status' => 0 // pending
        ]);

        return back()->with('success', 'Join request sent');
    }
    public function removeRequest(Club $club)
    {
        ClubMember::where('club_id', $club->id)
            ->where('student_id', auth()->id())
            ->where('status', 0)
            ->delete();

        return back()->with('success', 'Join request removed');
    }
    public function leave(Club $club)
    {
        ClubMember::where('club_id', $club->id)
            ->where('student_id', auth()->id())
            ->where('status', 1)
            ->delete();

        return back()->with('success', 'You left the club');
    }
    public function notice_index($clubId)
    {
        $club = Club::findOrFail($clubId);

        $notices = ClubNotice::where('club_id', $clubId)
            ->orderBy('notice_date', 'desc')
            ->get();

        return view('student.club.notice', compact('club', 'notices'));
    }
    public function notice_store(Request $request)
    {
        $request->validate([
            'club_id' => 'required',
            'title' => 'required',
            'description' => 'required',
            'notice_date' => 'required|date'
        ]);

        ClubNotice::create($request->all());

        return redirect()->back()->with('success', 'Notice created successfully');
    }
}
